#
# Copyright 2018-2019 IBM Corp. All Rights Reserved.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
#

from maxfw.core import MAX_API, PredictAPI
from core.model import ModelWrapper

from flask import abort
from flask_restplus import fields
from werkzeug.datastructures import FileStorage


# Set up parser for image input data
image_parser = MAX_API.parser()
image_parser.add_argument('image', type=FileStorage, location='files', required=True, help="An image file")

label_prediction = MAX_API.model('LabelPrediction', {
    'index': fields.String(required=False, description='Labels ranked by highest probability'),
    'caption': fields.String(required=True, description='Caption generated by image'),
    'probability': fields.Float(required=True, description="Probability of the caption")
})

predict_response = MAX_API.model('ModelPredictResponse', {
    'status': fields.String(required=True, description='Response status message'),
    'predictions': fields.List(fields.Nested(label_prediction), description='Predicted captions and probabilities')
})


class ModelPredictAPI(PredictAPI):

    model_wrapper = ModelWrapper()

    @MAX_API.doc('predict')
    @MAX_API.expect(image_parser)
    @MAX_API.marshal_with(predict_response)
    def post(self):
        """Make a prediction given input data"""

        result = {'status': 'error'}
        args = image_parser.parse_args()
        if not args['image'].mimetype.endswith(('jpg', 'jpeg', 'png')):
            abort(400, 'Invalid file type/extension. Please provide an image in JPEG or PNG format.')
        image_data = args['image'].read()

        preds = self.model_wrapper.predict(image_data)

        label_preds = [{'index': p[0], 'caption': p[1], 'probability': p[2]} for p in [x for x in preds]]
        result['predictions'] = label_preds
        result['status'] = 'ok'

        return result
